/**
 * \file caam_op_sign_verify.c
 *
 * \brief Architecture specific implementation of functions relevant for sign and verify
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <string.h>
#include "caam.h"
#include <linux/caam_ee.h>
#include <sdc/arch/errors.h>

#define CAAM_EE_MIN_TAG_LEN 8

#ifdef CAAM_EE_FLAG_INTERLEAVED_OP_BIT
    #define CAAM_EE_SIGN_DEFAULT_FLAGS CAAM_EE_FLAG_INTERLEAVED_OP_BIT
#else
    #define CAAM_EE_SIGN_DEFAULT_FLAGS 0
#endif

#define CAAM_EE_INIT_UPDATE_FINALIZE_FLAG 0  /* No interleaving mode is used */

/* define default type for sign/verify in IMX6 */
static const sdc_sign_verify_type_t sdc_sign_verify_default_type =
{SDC_SIGNVER_ALG_HMAC, SDC_SIGNVER_HASH_SHA256, SDC_NO_OPT_BMSK, CAAM_EE_SIGN_ALG_HMAC_FLAG, CAAM_EE_SIGN_ALG_HMAC_IDX, CAAM_EE_HASH_SHA256_FLAG, CAAM_EE_HASH_SHA256_IDX};


/* defined in sdc_arch.h */
const sdc_sign_verify_type_t *sdc_arch_sign_verify_get_default(void)
{
    return &sdc_sign_verify_default_type;
}


/* defined in sdc_arch.h */
sdc_error_t sdc_arch_sign_verify_type_alloc(sdc_sign_verify_type_t **type)
{
    *type = malloc(sizeof(sdc_sign_verify_type_t));
    if (!(*type))
        return SDC_NO_MEM;

    /* initialize with default */
    memcpy(*type, &sdc_sign_verify_default_type, sizeof(sdc_sign_verify_type_t));

    return SDC_OK;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_sign_verify_type_free(sdc_sign_verify_type_t *type)
{
    free (type);

    return SDC_OK;
}

static sdc_error_t caam_map_sdc_sign_verify_alg (sdc_sign_verify_alg_t alg, uint32_t *flag, uint32_t *idx, sdc_sign_verify_hash_t *default_hash)
{
    sdc_error_t err = SDC_OK;

    switch(alg) {
    case SDC_SIGNVER_ALG_HMAC:
        *flag = CAAM_EE_SIGN_ALG_HMAC_FLAG;
        *idx =  CAAM_EE_SIGN_ALG_HMAC_IDX;
        *default_hash = SDC_SIGNVER_HASH_SHA256;
        break;
    case SDC_SIGNVER_ALG_RSA:
        *flag = CAAM_EE_SIGN_ALG_RSA_PKCS1V15_FLAG;
        *idx =  CAAM_EE_SIGN_ALG_RSA_PKCS1V15_IDX;
        *default_hash = SDC_SIGNVER_HASH_SHA256;
        break;
    default:
        err = SDC_ALG_MODE_INVALID;
    }

    return err;
}
static sdc_error_t caam_map_sdc_sign_verify_hash (sdc_sign_verify_hash_t hash, uint32_t *flag, uint32_t *idx)
{
    sdc_error_t err = SDC_OK;

    switch(hash) {
    case SDC_SIGNVER_HASH_MD5:
        *flag = CAAM_EE_HASH_MD5_FLAG;
        *idx =  CAAM_EE_HASH_MD5_IDX;
        break;
    case SDC_SIGNVER_HASH_SHA1:
        *flag = CAAM_EE_HASH_SHA1_FLAG;
        *idx =  CAAM_EE_HASH_SHA1_IDX;
        break;
    case SDC_SIGNVER_HASH_SHA224:
        *flag = CAAM_EE_HASH_SHA224_FLAG;
        *idx =  CAAM_EE_HASH_SHA224_IDX;
        break;
    case SDC_SIGNVER_HASH_SHA256:
        *flag = CAAM_EE_HASH_SHA256_FLAG;
        *idx =  CAAM_EE_HASH_SHA256_IDX;
        break;
    default:
        err = SDC_ALG_MODE_INVALID;
    }

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_sign_verify_type_set_alg(sdc_sign_verify_type_t *type, sdc_sign_verify_alg_t alg)
{
    sdc_error_t err = SDC_OK;
    sdc_sign_verify_type_t tmp;

    memset(&tmp, 0, sizeof(sdc_sign_verify_type_t));

    tmp.alg = alg;
    err = caam_map_sdc_sign_verify_alg(tmp.alg, &tmp.arch_alg_flag,
                                       &tmp.arch_alg_idx, &tmp.hash);

    if (err == SDC_OK) {
        err = sdc_arch_sign_verify_type_set_hash (&tmp, tmp.hash);
    }

    if (err == SDC_OK) {
        memcpy(type, &tmp, sizeof(tmp));
    }

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_sign_verify_type_set_hash(sdc_sign_verify_type_t *type, sdc_sign_verify_hash_t hash)
{
    sdc_error_t err = SDC_OK;
    uint32_t hash_flag;
    uint32_t hash_idx;
    uint32_t alg_idx;

    err = caam_map_sdc_sign_verify_hash(hash, &hash_flag, &hash_idx);

    if (err == SDC_OK) {
        alg_idx = type->arch_alg_idx;

        /* invalid block mode */
        err = SDC_ALG_MODE_INVALID;

        /* check alg index - maybe someone has written directly to container */
        if (alg_idx < sizeof(caam_ee_sign_alg_descs) / sizeof(struct caam_ee_sign_alg_desc)) {
            /* check if hash is supported */
            if ((caam_ee_sign_alg_descs[alg_idx].hash_mode_msk & (1<<hash_idx)) != 0) {
                type->hash = hash;
                type->arch_hash_flag = hash_flag;
                type->arch_hash_idx = hash_idx;

                /* reset opt bmsk*/
                type->opt_bmsk = SDC_NO_OPT_BMSK;

                err = SDC_OK;
            }
        }
    }

    return err;
}

sdc_error_t sdc_arch_sign_verify_type_set_opt_bmsk(sdc_sign_verify_type_t *type, uint64_t opt_bmsk)
{
    /* we only support pre-computed hash option for RSA */

    if (type->alg != SDC_SIGNVER_ALG_RSA)
        return SDC_OP_NOT_SUPPORTED;

    if ((opt_bmsk != SDC_SIGNVER_OPT_PRECOMP_HASH) &&
        (opt_bmsk != SDC_NO_OPT_BMSK))
        return SDC_OP_NOT_SUPPORTED;

    type->opt_bmsk = opt_bmsk;

    return SDC_OK;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_sign_verify_type_get_alg(const sdc_sign_verify_type_t *type, sdc_sign_verify_alg_t *akg)
{

    *akg = type->alg;

    return SDC_OK;

}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_sign_verify_type_get_hash(const sdc_sign_verify_type_t *type, sdc_sign_verify_hash_t *hash)
{
    *hash = type->hash;

    return SDC_OK;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_sign_verify_type_get_opt_bmsk(const sdc_sign_verify_type_t *type, uint64_t *opt_bmsk)
{
    *opt_bmsk = type->opt_bmsk;

    return SDC_OK;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_sign_verify_key_desc_fill (
    const sdc_sign_verify_type_t *type,
    sdc_key_desc_t *desc)
{
    uint32_t alg_idx;
    uint32_t hash_idx;

    hash_idx = type->arch_hash_idx;
    alg_idx = type->arch_alg_idx;

    /* check range of alg */
    if (alg_idx >= sizeof(caam_ee_sign_alg_descs) / sizeof(struct caam_ee_sign_alg_desc))
        return SDC_ALG_MODE_INVALID;

    /* check if block mode is supported */
    if ((caam_ee_sign_alg_descs[alg_idx].hash_mode_msk & (1<<hash_idx)) == 0)
        return SDC_ALG_MODE_INVALID;

    caam_keylen_bmsk_to_sdc(caam_ee_sign_alg_descs[alg_idx].key_len_msk,
                            &(desc->sup_key_lens), &(desc->dflt_key_len));

    if (alg_idx == CAAM_EE_SIGN_ALG_RSA_PKCS1V15_IDX) {
        desc->sup_key_fmt_protect = SDC_KEY_FMT_RSA_PRIVATE;
        desc->sup_key_fmt_unprotect = SDC_KEY_FMT_RSA_PUBLIC;
    } else {
        desc->sup_key_fmt_protect = SDC_KEY_FMT_SIMPLE_SYM;
        desc->sup_key_fmt_unprotect = SDC_KEY_FMT_SIMPLE_SYM;
    }

    return SDC_OK;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_sign_verify_desc_fill (
    sdc_session_t *session,
    const sdc_sign_verify_type_t *type,
    sdc_sign_verify_desc_t *desc)
{
    sdc_error_t err = SDC_OK;
    const struct caam_ee_hash_desc *caam_ee_sign_hash_desc;
    uint32_t hash_idx;
    uint32_t alg_idx;
    sdc_key_info_t key_info;

    size_t min_data_len;
    size_t max_data_len;

    hash_idx = type->arch_hash_idx;
    alg_idx = type->arch_alg_idx;

    /* check range of alg */
    if (alg_idx >= sizeof(caam_ee_sign_alg_descs) / sizeof(struct caam_ee_sign_alg_desc))
        return SDC_ALG_MODE_INVALID;

    /* check if block mode is supported */
    if ((caam_ee_sign_alg_descs[alg_idx].hash_mode_msk & (1<<hash_idx)) == 0)
        return SDC_ALG_MODE_INVALID;

    caam_ee_sign_hash_desc = &caam_ee_hash_modes_descs[hash_idx];

    desc->iv.max = 0;
    desc->iv.min = 0;
    desc->iv.mod = 0;
    desc->iv.dflt = desc->iv.max;

    desc->data.block_len = caam_ee_sign_hash_desc->block_size;
    desc->data.max_chunk_len = caam_ee_sign_hash_desc->max_block_len;
    desc->data.chunk_len_aligned = true;
    min_data_len = 0;
    max_data_len = caam_ee_sign_hash_desc->max_total_len;

    desc->data.padding = SDC_PADDING_NO;
    if (caam_ee_sign_hash_desc->final_block_aligned) {
        desc->data.padding = SDC_PADDING_PKCS7;
    }
    desc->supports_iuf = true; /* default */

    if (alg_idx == CAAM_EE_SIGN_ALG_RSA_PKCS1V15_IDX) {
        /* session is mandatory - we need to read the key length */
        if (!session)
            err = SDC_SESSION_INVALID;

        if (err == SDC_OK) {
            err = sdc_arch_session_key_info_get(session, &key_info, false);

            if ((err == SDC_OK) &&
                (key_info.key_fmt != SDC_KEY_FMT_RSA_PRIVATE) &&
                (key_info.key_fmt != SDC_KEY_FMT_RSA_PUBLIC))
                err = SDC_KEY_FMT_INVALID;

            if (err == SDC_OK) {
                desc->tag.max = key_info.key_len_bytes;
                desc->tag.min = key_info.key_len_bytes;
                desc->tag.dflt = key_info.key_len_bytes;
                desc->tag.mod = 0;
            }

        }

        if ((err == SDC_OK) &&
            (type->opt_bmsk == SDC_SIGNVER_OPT_PRECOMP_HASH)) {
            min_data_len = caam_ee_sign_hash_desc->len_max;
            max_data_len = min_data_len;
            desc->supports_iuf = false; /* precalc hash only supports oneshot */
        }
    } else {
        /* optional check of key fmt */
        if (session) {
            err = sdc_arch_session_key_info_get(session, &key_info, false);

            if ((err == SDC_OK) &&
                (key_info.key_fmt != SDC_KEY_FMT_SIMPLE_SYM))
                err = SDC_KEY_FMT_INVALID;
        }

        /* no opt_bmsk supported for symmetric */
        if ((err == SDC_OK) && (type->opt_bmsk != SDC_NO_OPT_BMSK))
            err = SDC_ALG_MODE_INVALID;

        if (err == SDC_OK) {
            desc->tag.max = caam_ee_sign_hash_desc->len_max;
            desc->tag.min = caam_ee_sign_hash_desc->len_min;
            desc->tag.mod = 0;
            desc->tag.dflt = desc->tag.min; /*is RFC2104 recommended min value */
        }

        /* double check and change default in userspace if different from RFC2104 */
        if (desc->tag.dflt < (desc->tag.max/2))
            desc->tag.dflt = desc->tag.max/2;
        if ((desc->tag.dflt < CAAM_EE_MIN_TAG_LEN) && (CAAM_EE_MIN_TAG_LEN <= desc->tag.max)) {
            desc->tag.dflt = CAAM_EE_MIN_TAG_LEN;
        }
    }

    if (err == SDC_OK) {
        /* calculate max and alignment for plain and cipher */
        err = sdc_intern_get_plain_cipher_spec(
            desc->data.padding,
            min_data_len,
            max_data_len,
            desc->data.block_len,
            &(desc->data.plain),
            NULL);
    }

    return err;
}


static sdc_error_t caam_sign_verify_no_opt(sdc_session_t *session,
                                           caam_sign_verify_op operation,
                                           const sdc_sign_verify_type_t *type,
                                           const sdc_sign_verify_desc_t *desc,
                                           const uint8_t *in_data,
                                           const size_t in_data_len,
                                           const uint8_t *tag_in_data, const size_t tag_in_len,
                                           uint8_t *tag_out_data, const size_t tag_out_len)
{
    sdc_error_t err = SDC_OK;
    uint8_t padding_buffer[CAAM_EE_MAX_SIGN_BLOCK_ALIGN];
    const uint8_t *in_data_ptr;
    size_t in_len_remaining;
    size_t buf_idx;
    size_t buf_len;
    size_t align;
    size_t stop_len;
    struct caam_ee_sign_verify_params iodata;
    int res;
    int res_errno;
    unsigned long int request;
    bool padding_ioctl;

    if (desc->data.block_len > CAAM_EE_MAX_SIGN_BLOCK_ALIGN) {
        /* buffer not sufficient to handle padding */
        return SDC_UNKNOWN_ERROR;
    }

    in_data_ptr = in_data;
    in_len_remaining = in_data_len;
    buf_idx = 0;

    /* Initialize to no padding required
     * If no padding is required the complete data is handled by the
     * while loop only */
    stop_len = 0;
    /* padding is required in case hash needs it and not already aligned */
    if (desc->data.padding != SDC_PADDING_NO)
        stop_len = desc->data.block_len - 1;

    iodata.flags = type->arch_alg_flag | type->arch_hash_flag | CAAM_EE_SIGN_DEFAULT_FLAGS;

    if (operation == SIGN) {
        request = CAAM_SIGN_DATA;
        iodata.tag = tag_out_data;
        iodata.tag_len = tag_out_len;
    } else {
        request = CAAM_VERIFY_DATA;
        iodata.tag = (uint8_t *)tag_in_data;
        iodata.tag_len = tag_in_len;
    }

    /* if the first block (without padding) or if more data than padding len */
    while (((in_len_remaining > stop_len) || ((stop_len == 0) && (buf_idx == 0))) && (err == SDC_OK)) {
        buf_len = in_len_remaining;

        if (buf_len > desc->data.max_chunk_len)
            buf_len = desc->data.max_chunk_len;

        in_len_remaining -= buf_len;

        iodata.flags &= ~CAAM_OP_MASK;

        if ((in_len_remaining == 0) && (stop_len == 0)) {
            /* SINGLE or MULTI_FINALIZE will only happen if no padding is required */

            /* final or init-final */
            if (buf_idx == 0) {
                iodata.flags |= CAAM_OP_SINGLE;
            } else {
                iodata.flags |= CAAM_OP_MULTI_FINALIZE;
            }
        } else {
            /* init or update */

            /* alignment */
            align = buf_len % desc->data.block_len;
            buf_len -= align;
            in_len_remaining += align;

            if (buf_idx == 0) {
                iodata.flags |= CAAM_OP_MULTI_INIT;
            } else {
                iodata.flags |= CAAM_OP_MULTI_UPDATE;
            }
        }

        iodata.in = (uint8_t*)in_data_ptr;
        iodata.in_len = buf_len;

        res = ioctl(session->arch.fd, request, &iodata);

        if (res == 0) {
            buf_idx++;
            in_data_ptr += buf_len;
        } else {
            res_errno = errno;

            /* ignore and retry after EAGAIN */
            if (res_errno != EAGAIN) {
                /* even in RSA case the asymmetric operation will only be done during final */
                err = error_from_ioctl_errno(res_errno, request, iodata.flags);
            }

            in_len_remaining += buf_len;
        }
    }
    if ((stop_len) && (err == SDC_OK)) {
        /* 0 <= in_len_remaining <= stop_len = block-alignment - 1 */
        if (in_len_remaining) {
            /* copy remaining_data to padding buffer */
            memcpy(padding_buffer, in_data_ptr, in_len_remaining);
        }
        err = sdc_intern_pad(desc->data.padding, padding_buffer, in_len_remaining, desc->data.block_len);

        iodata.flags &= ~CAAM_OP_MASK;
        if (buf_idx == 0) {
            iodata.flags |= CAAM_OP_SINGLE;
        } else {
            iodata.flags |= CAAM_OP_MULTI_FINALIZE;
        }
        iodata.in = padding_buffer;
        iodata.in_len = desc->data.block_len;

        padding_ioctl = true;
        while ((padding_ioctl) && (err == SDC_OK)) {
            res = ioctl(session->arch.fd, request, &iodata);
            if (res != 0) {
                res_errno = errno;

                /* ignore and retry after EAGAIN */
                if (res_errno != EAGAIN) {
                    err = error_from_ioctl_errno(res_errno, request, iodata.flags);
                }
            } else {
                padding_ioctl = false;
            }
        }
    }

    sdc_intern_overwrite_secret(padding_buffer, CAAM_EE_MAX_SIGN_BLOCK_ALIGN);

    return err;
}

static sdc_error_t caam_sign_verify_with_opt(sdc_session_t *session,
                                             caam_sign_verify_op operation,
                                             const sdc_sign_verify_type_t *type,
                                             const sdc_sign_verify_desc_t *desc,
                                             const uint8_t *in_data,
                                             const size_t in_data_len,
                                             const uint8_t *tag_in_data, const size_t tag_in_len,
                                             uint8_t *tag_out_data, const size_t tag_out_len)
{
    sdc_error_t err = SDC_OK;
    int res;
    int res_errno;
    unsigned long int request;
    struct caam_ee_asym_sign_verify_hash_params iodata;

    /* only RSA precomputed hash supported at the moment */
    if ((type->alg != SDC_SIGNVER_ALG_RSA) ||
        (type->opt_bmsk != SDC_SIGNVER_OPT_PRECOMP_HASH))
        return SDC_ALG_MODE_INVALID;

    (void)desc;
    iodata.hash = (uint8_t*)in_data;
    iodata.hash_len = in_data_len;
    iodata.flags = type->arch_hash_flag;
    if (operation == SIGN) {
        request = CAAM_ASYM_SIGN_HASH;
        iodata.signature = tag_out_data;
        iodata.signature_len = tag_out_len;
    } else {
        request = CAAM_ASYM_VERIFY_HASH;
        iodata.signature = (uint8_t *)tag_in_data;
        iodata.signature_len = tag_in_len;
    }

    res = ioctl(session->arch.fd, request, &iodata);
    if (res != 0) {
        res_errno = errno;
        err = error_from_ioctl_errno(res_errno, request, iodata.flags);
    }

    return err;
}
static sdc_error_t caam_sign_verify(sdc_session_t *session,
                                    caam_sign_verify_op operation,
                                    const sdc_sign_verify_type_t *type,
                                    const sdc_sign_verify_desc_t *desc,
                                    const uint8_t *in_data,
                                    const size_t in_data_len,
                                    const uint8_t *tag_in_data, const size_t tag_in_len,
                                    uint8_t *tag_out_data, const size_t tag_out_len)
{
    sdc_error_t err = SDC_OK;

    if (caam_is_op_in_progress(session)) {
        /* Cancel non-completed operation if any */
        err = caam_perform_abort_operation(session);
        if (err != SDC_OK)
            return err;
    }

    if (type->opt_bmsk == 0) {
        err = caam_sign_verify_no_opt(session, operation,
                                      type, desc,
                                      in_data, in_data_len,
                                      tag_in_data, tag_in_len,
                                      tag_out_data, tag_out_len);
    } else {
        err = caam_sign_verify_with_opt(session, operation,
                                        type, desc,
                                        in_data, in_data_len,
                                        tag_in_data, tag_in_len,
                                        tag_out_data, tag_out_len);
    }

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_sign(sdc_session_t *session,
                          const sdc_sign_verify_type_t *type,
                          const sdc_sign_verify_desc_t *desc,
                          const uint8_t *in_data,
                          const size_t in_data_len,
                          const uint8_t *iv, const size_t iv_len,
                          uint8_t *tag_out_data, const size_t tag_out_len)
{
    (void)iv;
    (void)iv_len;

    return caam_sign_verify (session, SIGN, type, desc, in_data,
                             in_data_len, NULL, 0, tag_out_data, tag_out_len);
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_verify(sdc_session_t *session,
                            const sdc_sign_verify_type_t *type,
                            const sdc_sign_verify_desc_t *desc,
                            const uint8_t *in_data,
                            const size_t in_data_len,
                            const uint8_t *iv, const size_t iv_len,
                            const uint8_t *tag_in_data, const size_t tag_in_len)
{
    (void)iv;
    (void)iv_len;

    return caam_sign_verify (session, VERIFY, type, desc, in_data,
                             in_data_len, tag_in_data, tag_in_len, NULL, 0);
}

static sdc_error_t caam_arch_sign_verify_init(sdc_session_t *session,
                                              const sdc_sign_verify_desc_t *desc,
                                              const uint8_t *iv, size_t iv_len, caam_sign_verify_op operation)
{

    sdc_error_t err = SDC_OK;

    if (desc->data.block_len > CAAM_EE_MAX_SIGN_BLOCK_ALIGN) {
        /* buffer not sufficient to handle padding */
        return SDC_INTERNAL_ERROR;
    }

    if (caam_is_op_in_progress(session)) {
        /* Cancel non-completed operation if any */
        err = caam_perform_abort_operation(session);
        if (err != SDC_OK)
            return err;
    }

    memcpy(session->arch.iv, (uint8_t *) iv, iv_len);
    session->arch.iv_len = (size_t) iv_len;

    if (operation == SIGN) {
        session->arch.request = CAAM_SIGN_DATA;
    } else {
        session->arch.request = CAAM_VERIFY_DATA;
    }

    session->arch.buf_idx = 0;

    return err;
}

static sdc_error_t caam_sign_verify_update(sdc_session_t *session,
                                           const sdc_sign_verify_desc_t *desc,
                                           const sdc_sign_verify_type_t *type,
                                           const uint8_t *in_data, const size_t in_data_len)
{
    sdc_error_t err = SDC_OK;
    struct caam_ee_sign_verify_params iodata;
    uint32_t iodata_flags = type->arch_alg_flag | type->arch_hash_flag | CAAM_EE_INIT_UPDATE_FINALIZE_FLAG;
    int res_errno = 0;
    int res;

    /* data needs to be aligned */
    if ((in_data_len % desc->data.block_len) != 0)
        err = SDC_INTERNAL_ERROR;

    if (err == SDC_OK) {
        if (session->arch.buf_idx == 0) {
            iodata_flags |= CAAM_OP_MULTI_INIT;
        } else {
            iodata_flags |= CAAM_OP_MULTI_UPDATE;
        }

        /* retry in case of EAGAIN */
        while (err == SDC_OK) {
            res = caam_perform_sign_verify_ioctl_operation(session, &iodata,
                                                           in_data, in_data_len,
                                                           NULL, 0, NULL, 0,
                                                           iodata_flags);
            res_errno = errno;

            if (res == 0) {
                break;
            } else {
                /* ignore and retry after EAGAIN */
                if (res_errno != EAGAIN) {
                    /* even in RSA case the asymmetric operation will only be done during final */
                    err = error_from_ioctl_errno(res_errno, session->arch.request, iodata_flags);
                }
            }
        }
        if (err == SDC_OK) {
            session->arch.buf_idx++;
        }
    }

    return err;
}

static sdc_error_t caam_sign_verify_finalize(sdc_session_t *session,
                                             const sdc_sign_verify_type_t *type,
                                             const sdc_sign_verify_desc_t *desc,
                                             const uint8_t *in_data, const size_t in_data_len,
                                             uint8_t *tag_out_data, size_t tag_out_data_len,
                                             const uint8_t *tag_in_data, size_t tag_in_data_len)
{
    sdc_error_t err = SDC_OK;
    struct caam_ee_sign_verify_params iodata;
    uint32_t iodata_flags = type->arch_alg_flag | type->arch_hash_flag | CAAM_EE_INIT_UPDATE_FINALIZE_FLAG;
    int res_errno = 0;
    int res;
    size_t block_len = desc->data.block_len;
    const uint8_t *io_in_ptr = in_data;
    size_t io_in_len = in_data_len;

    /* in data must not exceed one block */
    if (in_data_len > block_len)
        err = SDC_INTERNAL_ERROR;

    if (err == SDC_OK) {
        if (session->arch.buf_idx == 0) {
            iodata_flags |= CAAM_OP_SINGLE;
        } else {
            iodata_flags |= CAAM_OP_MULTI_FINALIZE;
        }

        /* retry in case of EAGAIN */
        while (err == SDC_OK) {
            res = caam_perform_sign_verify_ioctl_operation(session, &iodata,
                                                           io_in_ptr, io_in_len,
                                                           tag_out_data,
                                                           tag_out_data_len,
                                                           tag_in_data,
                                                           tag_in_data_len,
                                                           iodata_flags);
            res_errno = errno;

            if (res == 0) {
                break;
            } else {
                /* ignore and retry after EAGAIN */
                if (res_errno != EAGAIN) {
                    err = error_from_ioctl_errno(res_errno, session->arch.request, iodata_flags);
                }
            }
        }
    }

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_sign_init(sdc_session_t *session,
                               const sdc_sign_verify_type_t *type,
                               const sdc_sign_verify_desc_t *desc,
                               const uint8_t *iv, size_t iv_len)
{
    (void)type;
    return caam_arch_sign_verify_init(session, desc, iv, iv_len, SIGN);
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_verify_init(sdc_session_t *session,
                                 const sdc_sign_verify_type_t *type,
                                 const sdc_sign_verify_desc_t *desc,
                                 const uint8_t *iv, size_t iv_len)
{
    (void)type;
    return caam_arch_sign_verify_init(session, desc, iv, iv_len, VERIFY);
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_sign_update(sdc_session_t *session,
                                 const sdc_sign_verify_type_t *type,
                                 const sdc_sign_verify_desc_t *desc,
                                 const uint8_t *in_data, const size_t in_data_len)
{
    return caam_sign_verify_update(session, desc, type, in_data, in_data_len);
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_verify_update(sdc_session_t *session,
                                   const sdc_sign_verify_type_t *type,
                                   const sdc_sign_verify_desc_t *desc,
                                   const uint8_t *in_data, const size_t in_data_len)
{
    return caam_sign_verify_update(session, desc, type, in_data, in_data_len);
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_sign_finalize(sdc_session_t *session,
                                   const sdc_sign_verify_type_t *type,
                                   const sdc_sign_verify_desc_t *desc,
                                   const uint8_t *in_data, const size_t in_data_len,
                                   uint8_t *tag_data, size_t tag_data_len)
{
    return caam_sign_verify_finalize(session, type, desc, in_data, in_data_len,
                                     tag_data, tag_data_len, NULL, 0);
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_verify_finalize(sdc_session_t *session,
                                     const sdc_sign_verify_type_t *type,
                                     const sdc_sign_verify_desc_t *desc,
                                     const uint8_t *in_data, const size_t in_data_len,
                                     const uint8_t *tag_data, size_t tag_data_len)
{
    return caam_sign_verify_finalize(session, type, desc, in_data, in_data_len,
                                     NULL, 0, tag_data, tag_data_len);
}
